/*
 * Copyright (C) 2016 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"

#include <string>
#include <wtf/LEBDecoder.h>
#include <wtf/Vector.h>

namespace TestWebKitAPI {

static std::string toString(const Vector<uint8_t>& vector)
{
    std::stringstream out;
    out << std::hex;
    out << "{ ";
    for (uint8_t v : vector)
        out << "0x" << std::setfill('0') << std::setw(2) << static_cast<unsigned>(v) << ", ";
    out << "}";
    return out.str();
}

static void testUInt32LEBDecode(std::initializer_list<uint8_t> data, size_t startOffset, bool expectedStatus, uint32_t expectedResult, size_t expectedOffset)
{
    Vector<uint8_t> vector(data);
    auto string = toString(vector);
    uint32_t result;
    bool status = WTF::LEBDecoder::decodeUInt32(vector.data(), vector.size(), startOffset, result);
    EXPECT_EQ(expectedStatus, status) << string;
    if (expectedStatus) {
        EXPECT_EQ(expectedResult, result) << string;
        EXPECT_EQ(expectedOffset, startOffset) << string;
    }
}

TEST(WTF, LEBDecoderUInt32)
{
    // Simple tests that use all the bits in the array
    testUInt32LEBDecode({ 0x07 }, 0, true, 0x7lu, 1lu);
    testUInt32LEBDecode({ 0x77 }, 0, true, 0x77lu, 1lu);
    testUInt32LEBDecode({ 0x80, 0x07 }, 0, true, 0x380lu, 2lu);
    testUInt32LEBDecode({ 0x89, 0x12 }, 0, true, 0x909lu, 2lu);
    testUInt32LEBDecode({ 0xf3, 0x85, 0x02 }, 0, true, 0x82f3lu, 3lu);
    testUInt32LEBDecode({ 0xf3, 0x85, 0xff, 0x74 }, 0, true, 0xe9fc2f3lu, 4lu);
    testUInt32LEBDecode({ 0xf3, 0x85, 0xff, 0xf4, 0x0f }, 0, true, 0xfe9fc2f3lu, 5lu);
    testUInt32LEBDecode({ 0xff, 0xff, 0xff, 0xff, 0x0f }, 0, true, 0xfffffffflu, 5lu);
    // Test with extra trailing numbers
    testUInt32LEBDecode({ 0x07, 0x80 }, 0, true, 0x7lu, 1lu);
    testUInt32LEBDecode({ 0x07, 0x75 }, 0, true, 0x7lu, 1lu);
    testUInt32LEBDecode({ 0xf3, 0x85, 0xff, 0x74, 0x43 }, 0, true, 0xe9fc2f3lu, 4lu);
    testUInt32LEBDecode({ 0xf3, 0x85, 0xff, 0x74, 0x80 }, 0, true, 0xe9fc2f3lu, 4lu);
    // Test with preceeding numbers
    testUInt32LEBDecode({ 0xf3, 0x07 }, 1, true, 0x7lu, 2lu);
    testUInt32LEBDecode({ 0x03, 0x07 }, 1, true, 0x7lu, 2lu);
    testUInt32LEBDecode({ 0xf2, 0x53, 0x43, 0x67, 0x79, 0x77 }, 5, true, 0x77lu, 6lu);
    testUInt32LEBDecode({ 0xf2, 0x53, 0x43, 0xf7, 0x84, 0x77 }, 5, true, 0x77lu, 6ul);
    testUInt32LEBDecode({ 0xf2, 0x53, 0x43, 0xf3, 0x85, 0x02 }, 3, true, 0x82f3lu, 6lu);
    // Test in the middle
    testUInt32LEBDecode({ 0xf3, 0x07, 0x89 }, 1, true, 0x7lu, 2lu);
    testUInt32LEBDecode({ 0x03, 0x07, 0x23 }, 1, true, 0x7lu, 2lu);
    testUInt32LEBDecode({ 0xf2, 0x53, 0x43, 0x67, 0x79, 0x77, 0x43 }, 5, true, 0x77lu, 6lu);
    testUInt32LEBDecode({ 0xf2, 0x53, 0x43, 0xf7, 0x84, 0x77, 0xf9 }, 5, true, 0x77lu, 6lu);
    testUInt32LEBDecode({ 0xf2, 0x53, 0x43, 0xf3, 0x85, 0x02, 0xa4 }, 3, true, 0x82f3lu, 6lu);
    // Test decode too long
    testUInt32LEBDecode({ 0x80, 0x80, 0x80, 0x80, 0x80, 0x80 }, 0, false, 0x0lu, 0lu);
    testUInt32LEBDecode({ 0x80, 0x80, 0xab, 0x8a, 0x9a, 0xa3, 0xff }, 1, false, 0x0lu, 0lu);
    testUInt32LEBDecode({ 0x80, 0x80, 0xab, 0x8a, 0x9a, 0xa3, 0xff }, 0, false, 0x0lu, 0lu);
    // Test decode off end of array
    testUInt32LEBDecode({ 0x80, 0x80, 0xab, 0x8a, 0x9a, 0xa3, 0xff }, 2, false, 0x0lu, 0lu);
    // Test decode overflow
    testUInt32LEBDecode({ 0xf3, 0x85, 0xff, 0xf4, 0x1f }, 0, false, 0x0lu, 0lu);
    testUInt32LEBDecode({ 0xff, 0xff, 0xff, 0xff, 0x10 }, 0, false, 0x0lu, 0lu);
}

static void testUInt64LEBDecode(std::initializer_list<uint8_t> data, size_t startOffset, bool expectedStatus, uint64_t expectedResult, size_t expectedOffset)
{
    Vector<uint8_t> vector(data);
    auto string = toString(vector);
    uint64_t result;
    bool status = WTF::LEBDecoder::decodeUInt64(vector.data(), vector.size(), startOffset, result);
    EXPECT_EQ(expectedStatus, status) << string;
    if (expectedStatus) {
        EXPECT_EQ(expectedResult, result) << string;
        EXPECT_EQ(expectedOffset, startOffset) << string;
    }
}

TEST(WTF, LEBDecoderUInt64)
{
    // Simple tests that use all the bits in the array
    testUInt64LEBDecode({ 0x07 }, 0, true, 0x7lu, 1lu);
    testUInt64LEBDecode({ 0x77 }, 0, true, 0x77lu, 1lu);
    testUInt64LEBDecode({ 0x80, 0x07 }, 0, true, 0x380lu, 2lu);
    testUInt64LEBDecode({ 0x89, 0x12 }, 0, true, 0x909lu, 2lu);
    testUInt64LEBDecode({ 0xf3, 0x85, 0x02 }, 0, true, 0x82f3lu, 3lu);
    testUInt64LEBDecode({ 0xf3, 0x85, 0xff, 0x74 }, 0, true, 0xe9fc2f3lu, 4lu);
    testUInt64LEBDecode({ 0xf3, 0x85, 0xff, 0xf4, 0x7f }, 0, true, 0x7fe9fc2f3lu, 5lu);
    testUInt64LEBDecode({ 0xf3, 0x85, 0xff, 0xf4, 0xff, 0x4b }, 0, true, 0x25ffe9fc2f3lu, 6lu);
    testUInt64LEBDecode({ 0xf3, 0x85, 0xff, 0xf4, 0xff, 0xcb, 0x3a }, 0, true, 0xea5ffe9fc2f3lu, 7lu);
    testUInt64LEBDecode({ 0xf3, 0x85, 0xff, 0xf4, 0xff, 0xcb, 0xba, 0x0f }, 0, true, 0x1eea5ffe9fc2f3lu, 8lu);
    testUInt64LEBDecode({ 0xf3, 0x85, 0xff, 0xf4, 0xff, 0xcb, 0xba, 0x8f, 0x69 }, 0, true, 0x691eea5ffe9fc2f3lu, 9lu);
    testUInt64LEBDecode({ 0xf3, 0x85, 0xff, 0xf4, 0xff, 0xcb, 0xba, 0x8f, 0xe9, 0x01 }, 0, true, 0xe91eea5ffe9fc2f3lu, 10lu);
    testUInt64LEBDecode({ 0xf3, 0x85, 0xff, 0xf4, 0xff, 0xcb, 0xba, 0x8f, 0xe9, 0x00 }, 0, true, 0x691eea5ffe9fc2f3lu, 10lu);
    testUInt64LEBDecode({ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x01 }, 0, true, 0xfffffffffffffffflu, 10lu);
    // Test with extra trailing numbers
    testUInt64LEBDecode({ 0x07, 0x80 }, 0, true, 0x7lu, 1lu);
    testUInt64LEBDecode({ 0x07, 0x75 }, 0, true, 0x7lu, 1lu);
    testUInt64LEBDecode({ 0xf3, 0x85, 0xff, 0x74, 0x43 }, 0, true, 0xe9fc2f3lu, 4lu);
    testUInt64LEBDecode({ 0xf3, 0x85, 0xff, 0x74, 0x80 }, 0, true, 0xe9fc2f3lu, 4lu);
    testUInt64LEBDecode({ 0xf3, 0x85, 0xff, 0xf4, 0xff, 0xcb, 0xba, 0x8f, 0x69, 0x45 }, 0, true, 0x691eea5ffe9fc2f3lu, 9lu);
    // Test with preceeding numbers
    testUInt64LEBDecode({ 0xf3, 0x07 }, 1, true, 0x7lu, 2lu);
    testUInt64LEBDecode({ 0x03, 0x07 }, 1, true, 0x7lu, 2lu);
    testUInt64LEBDecode({ 0xf2, 0x53, 0x43, 0x67, 0x79, 0x77 }, 5, true, 0x77lu, 6lu);
    testUInt64LEBDecode({ 0xf2, 0x53, 0x43, 0xf7, 0x84, 0x77 }, 5, true, 0x77lu, 6ul);
    testUInt64LEBDecode({ 0xf2, 0x53, 0x43, 0xf3, 0x85, 0x02 }, 3, true, 0x82f3lu, 6lu);
    testUInt64LEBDecode({ 0x92, 0xf3, 0x85, 0xff, 0xf4, 0xff, 0xcb, 0xba, 0x8f, 0x69 }, 1, true, 0x691eea5ffe9fc2f3lu, 10lu);
    // Test in the middle
    testUInt64LEBDecode({ 0xf3, 0x07, 0x89 }, 1, true, 0x7lu, 2lu);
    testUInt64LEBDecode({ 0x03, 0x07, 0x23 }, 1, true, 0x7lu, 2lu);
    testUInt64LEBDecode({ 0xf2, 0x53, 0x43, 0x67, 0x79, 0x77, 0x43 }, 5, true, 0x77lu, 6lu);
    testUInt64LEBDecode({ 0xf2, 0x53, 0x43, 0xf7, 0x84, 0x77, 0xf9 }, 5, true, 0x77lu, 6lu);
    testUInt64LEBDecode({ 0xf2, 0x53, 0x43, 0xf3, 0x85, 0x02, 0xa4 }, 3, true, 0x82f3lu, 6lu);
    testUInt64LEBDecode({ 0x92, 0xf3, 0x85, 0xff, 0xf4, 0xff, 0xcb, 0xba, 0x8f, 0x69, 0x85, 0x75 }, 1, true, 0x691eea5ffe9fc2f3lu, 10lu);
    testUInt64LEBDecode({ 0x92, 0x65, 0xf3, 0x85, 0xff, 0xf4, 0xff, 0xcb, 0xba, 0x8f, 0x69, 0x85, 0x75 }, 2, true, 0x691eea5ffe9fc2f3lu, 11lu);
    // Test decode too long
    testUInt64LEBDecode({ 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80 }, 0, false, 0x0lu, 0lu);
    testUInt64LEBDecode({ 0x80, 0x80, 0xab, 0x8a, 0x9a, 0xa3, 0xa3, 0x9f, 0xd2, 0xef, 0x8a, 0x4e }, 1, false, 0x0lu, 0lu);
    testUInt64LEBDecode({ 0x80, 0x80, 0xab, 0x8a, 0x9a, 0xa3, 0xff, 0xef, 0xd8, 0xee, 0xaa, 0xbb }, 0, false, 0x0lu, 0lu);
    testUInt64LEBDecode({ 0xf3, 0x85, 0xff, 0xf4, 0xff, 0xcb, 0xba, 0x8f, 0xa9, 0xa8, 0x05 }, 0, false, 0x0lu, 0lu);
    // Test decode off end of array
    testUInt64LEBDecode({ 0x80, 0x80, 0xab, 0x8a, 0x9a, 0xa3, 0xff }, 2, false, 0x0lu, 0lu);
    testUInt64LEBDecode({ 0x80, 0x80, 0xab, 0x8a, 0x9a, 0xa3, 0xff }, 2, false, 0x0lu, 0lu);
    testUInt64LEBDecode({ 0x92, 0xf3, 0x85, 0xff, 0xf4, 0xff, 0xcb, 0xba, 0x8f }, 1, false, 0x0lu, 0lu);
    // Test decode overflow
    testUInt64LEBDecode({ 0xf3, 0x85, 0xff, 0xf4, 0xff, 0xcb, 0xba, 0x8f, 0xe9, 0x02 }, 0, false, 0x0lu, 0lu);
}

static void testInt32LEBDecode(std::initializer_list<uint8_t> data, size_t startOffset, bool expectedStatus, int32_t expectedResult, size_t expectedOffset)
{
    Vector<uint8_t> vector(data);
    auto string = toString(vector);
    int32_t result;
    bool status = WTF::LEBDecoder::decodeInt32(vector.data(), vector.size(), startOffset, result);
    EXPECT_EQ(expectedStatus, status) << string;
    if (expectedStatus) {
        EXPECT_EQ(expectedResult, result) << string;
        EXPECT_EQ(expectedOffset, startOffset) << string;
    }
}

TEST(WTF, LEBDecoderInt32)
{
    // Simple tests that use all the bits in the array
    testInt32LEBDecode({ 0x07 }, 0, true, 0x7, 1lu);
    testInt32LEBDecode({ 0x77 }, 0, true, -0x9, 1lu);
    testInt32LEBDecode({ 0x80, 0x07 }, 0, true, 0x380, 2lu);
    testInt32LEBDecode({ 0x89, 0x12 }, 0, true, 0x909, 2lu);
    testInt32LEBDecode({ 0xf3, 0x85, 0x02 }, 0, true, 0x82f3, 3lu);
    testInt32LEBDecode({ 0xf3, 0x85, 0xff, 0x74 }, 0, true, 0xfe9fc2f3, 4lu);
    testInt32LEBDecode({ 0xf3, 0x85, 0xff, 0xf4, 0x7f }, 0, true, 0xfe9fc2f3, 5lu);
    testInt32LEBDecode({ 0xff, 0xff, 0xff, 0xff, 0x07 }, 0, true, INT32_MAX, 5lu);
    testInt32LEBDecode({ 0xff, 0xff, 0xff, 0xff, 0x7f }, 0, true, -1, 5lu);
    testInt32LEBDecode({ 0xff, 0xff, 0xff, 0xff, 0x7b }, 0, true, -1073741825, 5lu);
    // Test with extra trailing numbers
    testInt32LEBDecode({ 0x07, 0x80 }, 0, true, 0x7, 1lu);
    testInt32LEBDecode({ 0x07, 0x75 }, 0, true, 0x7, 1lu);
    testInt32LEBDecode({ 0xf3, 0x85, 0xff, 0x74, 0x43 }, 0, true, 0xfe9fc2f3, 4lu);
    testInt32LEBDecode({ 0xf3, 0x85, 0xff, 0x74, 0x80 }, 0, true, 0xfe9fc2f3, 4lu);
    // Test with preceeding numbers
    testInt32LEBDecode({ 0xf3, 0x07 }, 1, true, 0x7, 2lu);
    testInt32LEBDecode({ 0x03, 0x07 }, 1, true, 0x7, 2lu);
    testInt32LEBDecode({ 0xf2, 0x53, 0x43, 0x67, 0x79, 0x77 }, 5, true, -0x9, 6lu);
    testInt32LEBDecode({ 0xf2, 0x53, 0x43, 0xf7, 0x84, 0x77 }, 5, true, -0x9, 6lu);
    testInt32LEBDecode({ 0xf2, 0x53, 0x43, 0xf3, 0x85, 0x02 }, 3, true, 0x82f3, 6lu);
    // Test in the middle
    testInt32LEBDecode({ 0xf3, 0x07, 0x89 }, 1, true, 0x7, 2lu);
    testInt32LEBDecode({ 0x03, 0x07, 0x23 }, 1, true, 0x7, 2lu);
    testInt32LEBDecode({ 0xf2, 0x53, 0x43, 0x67, 0x79, 0x77, 0x43 }, 5, true, -0x9, 6lu);
    testInt32LEBDecode({ 0xf2, 0x53, 0x43, 0xf7, 0x84, 0x77, 0xf9 }, 5, true, -0x9, 6lu);
    testInt32LEBDecode({ 0xf2, 0x53, 0x43, 0xf3, 0x85, 0x02, 0xa4 }, 3, true, 0x82f3, 6lu);
    // Test decode too long
    testInt32LEBDecode({ 0x80, 0x80, 0x80, 0x80, 0x80, 0x80 }, 0, false, 0x0, 0lu);
    testInt32LEBDecode({ 0x80, 0x80, 0xab, 0x8a, 0x9a, 0xa3, 0xff }, 1, false, 0x0, 0lu);
    testInt32LEBDecode({ 0x80, 0x80, 0xab, 0x8a, 0x9a, 0xa3, 0xff }, 0, false, 0x0, 0lu);
    // Test decode off end of array
    testInt32LEBDecode({ 0x80, 0x80, 0xab, 0x8a, 0x9a, 0xa3, 0xff }, 2, false, 0x0, 0lu);
    // Test decode overflow
    testInt32LEBDecode({ 0xff, 0xff, 0xff, 0xff, 0x08 }, 0, false, 0, 0lu);
    testInt32LEBDecode({ 0xff, 0xff, 0xff, 0xff, 0x77 }, 0, false, 0, 0lu);
}

static void testInt64LEBDecode(std::initializer_list<uint8_t> data, size_t startOffset, bool expectedStatus, int64_t expectedResult, size_t expectedOffset)
{
    Vector<uint8_t> vector(data);
    auto string = toString(vector);
    int64_t result;
    bool status = WTF::LEBDecoder::decodeInt64(vector.data(), vector.size(), startOffset, result);
    EXPECT_EQ(expectedStatus, status) << string;
    if (expectedStatus) {
        EXPECT_EQ(expectedResult, result) << string;
        EXPECT_EQ(expectedOffset, startOffset) << string;
    }
}

TEST(WTF, LEBDecoderInt64)
{
    // Simple tests that use all the bits in the array
    testInt64LEBDecode({ 0x07 }, 0, true, 0x7, 1lu);
    testInt64LEBDecode({ 0x77 }, 0, true, -0x9, 1lu);
    testInt64LEBDecode({ 0x80, 0x07 }, 0, true, 0x380, 2lu);
    testInt64LEBDecode({ 0x89, 0x12 }, 0, true, 0x909, 2lu);
    testInt64LEBDecode({ 0xf3, 0x85, 0x02 }, 0, true, 0x82f3, 3lu);
    testInt64LEBDecode({ 0xf3, 0x85, 0xff, 0x74 }, 0, true, 0xfffffffffe9fc2f3, 4lu);
    testInt64LEBDecode({ 0xf3, 0x85, 0xff, 0xf4, 0x7f }, 0, true, 0xfffffffffe9fc2f3, 5lu);
    testInt64LEBDecode({ 0xf3, 0x85, 0xff, 0xf4, 0x3f }, 0, true, 0x3fe9fc2f3, 5lu);
    testInt64LEBDecode({ 0xf3, 0x85, 0xff, 0xf4, 0x8f, 0x1a }, 0, true, 0xd0fe9fc2f3, 6lu);
    testInt64LEBDecode({ 0xf3, 0x85, 0xff, 0xf4, 0x8f, 0x9a, 0x80, 0x2a }, 0, true, 0x5400d0fe9fc2f3, 8lu);
    testInt64LEBDecode({ 0xf3, 0x85, 0xff, 0xf4, 0x8f, 0x9a, 0x80, 0xaa, 0x41 }, 0, true, 0xc15400d0fe9fc2f3, 9lu);
    testInt64LEBDecode({ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x3f, }, 0, true, INT64_MAX >> 1, 9lu);
    testInt64LEBDecode({ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, }, 0, true, -1, 9lu);
    testInt64LEBDecode({ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00 }, 0, true, INT64_MAX, 10lu);
    testInt64LEBDecode({ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f }, 0, true, -1, 10lu);
    testInt64LEBDecode({ 0xf3, 0x85, 0xff, 0xf4, 0x8f, 0x9a, 0x80, 0xaa, 0xc1, 0x7f }, 0, true, 0xc15400d0fe9fc2f3, 10lu);
    testInt64LEBDecode({ 0xf3, 0x85, 0xff, 0xf4, 0x8f, 0x9a, 0x80, 0xaa, 0xc1, 0x00 }, 0, true, 0x415400d0fe9fc2f3, 10lu);
    // Test with extra trailing numbers
    testInt64LEBDecode({ 0x07, 0x80 }, 0, true, 0x7, 1lu);
    testInt64LEBDecode({ 0x07, 0x75 }, 0, true, 0x7, 1lu);
    testInt64LEBDecode({ 0xf3, 0x85, 0xff, 0x74, 0x43 }, 0, true, 0xfffffffffe9fc2f3, 4lu);
    testInt64LEBDecode({ 0xf3, 0x85, 0xff, 0x74, 0x80 }, 0, true, 0xfffffffffe9fc2f3, 4lu);
    // Test with preceeding numbers
    testInt64LEBDecode({ 0xf3, 0x07 }, 1, true, 0x7, 2lu);
    testInt64LEBDecode({ 0x03, 0x07 }, 1, true, 0x7, 2lu);
    testInt64LEBDecode({ 0xf2, 0x53, 0x43, 0x67, 0x79, 0x77 }, 5, true, -0x9, 6lu);
    testInt64LEBDecode({ 0xf2, 0x53, 0x43, 0xf7, 0x84, 0x77 }, 5, true, -0x9, 6lu);
    testInt64LEBDecode({ 0xf2, 0x53, 0x43, 0xf3, 0x85, 0x02 }, 3, true, 0x82f3, 6lu);
    // Test in the middle
    testInt64LEBDecode({ 0xf3, 0x07, 0x89 }, 1, true, 0x7, 2lu);
    testInt64LEBDecode({ 0x03, 0x07, 0x23 }, 1, true, 0x7, 2lu);
    testInt64LEBDecode({ 0xf2, 0x53, 0x43, 0x67, 0x79, 0x77, 0x43 }, 5, true, -0x9, 6lu);
    testInt64LEBDecode({ 0xf2, 0x53, 0x43, 0xf7, 0x84, 0x77, 0xf9 }, 5, true, -0x9, 6lu);
    testInt64LEBDecode({ 0xf2, 0x53, 0x43, 0xf3, 0x85, 0x02, 0xa4 }, 3, true, 0x82f3, 6lu);
    // Test decode too long
    testInt64LEBDecode({ 0x80, 0x80, 0x80, 0x80, 0x80, 0x80 }, 0, false, 0x0, 0lu);
    testInt64LEBDecode({ 0x80, 0x80, 0xab, 0x8a, 0x9a, 0xa3, 0xff }, 1, false, 0x0, 0lu);
    testInt64LEBDecode({ 0x80, 0x80, 0xab, 0x8a, 0x9a, 0xa3, 0xff }, 0, false, 0x0, 0lu);
    testInt64LEBDecode({ 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x00 }, 0, false, 0lu, 0lu);
    // Test decode off end of array
    testInt64LEBDecode({ 0x80, 0x80, 0xab, 0x8a, 0x9a, 0xa3, 0xff }, 2, false, 0x0, 0lu);
    // Test decode overflow
    testInt64LEBDecode({ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x01 }, 0, false, 0, 0lu);
    testInt64LEBDecode({ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7e }, 0, false, 0, 0lu);
}


} // namespace TestWebKitAPI
