/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This file is part of the LibreOffice project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * This file incorporates work covered by the following license notice:
 *
 *   Licensed to the Apache Software Foundation (ASF) under one or more
 *   contributor license agreements. See the NOTICE file distributed
 *   with this work for additional information regarding copyright
 *   ownership. The ASF licenses this file to you under the Apache
 *   License, Version 2.0 (the "License"); you may not use this file
 *   except in compliance with the License. You may obtain a copy of
 *   the License at http://www.apache.org/licenses/LICENSE-2.0 .
 */

#include "xmlbas_import.hxx"
#include "xmlscript/xmlns.h"
#include "xmlscript/xml_helper.hxx"
#include <com/sun/star/beans/XPropertySet.hpp>
#include <com/sun/star/lang/XMultiComponentFactory.hpp>
#include <com/sun/star/script/XLibraryContainerPassword.hpp>
#include <com/sun/star/document/XEmbeddedScripts.hpp>
#include <cppuhelper/implementationentry.hxx>

using namespace ::com::sun::star;
using namespace ::com::sun::star::lang;
using namespace ::com::sun::star::uno;


//.........................................................................
namespace xmlscript
{
//.........................................................................

    // =============================================================================
    // BasicElementBase
    // =============================================================================

    BasicElementBase::BasicElementBase( const OUString& rLocalName,
            const Reference< xml::input::XAttributes >& xAttributes,
            BasicElementBase* pParent, BasicImport* pImport )
        :m_pImport( pImport )
        ,m_pParent( pParent )
        ,m_aLocalName( rLocalName )
        ,m_xAttributes( xAttributes )
    {
        if ( m_pImport )
            m_pImport->acquire();
        if ( m_pParent )
            m_pParent->acquire();
    }

    // -----------------------------------------------------------------------------

    BasicElementBase::~BasicElementBase()
    {
        if ( m_pImport )
            m_pImport->release();
        if ( m_pParent )
            m_pParent->release();
    }

    // -----------------------------------------------------------------------------

    bool BasicElementBase::getBoolAttr( sal_Bool* pRet, const OUString& rAttrName,
        const ::com::sun::star::uno::Reference< ::com::sun::star::xml::input::XAttributes >& xAttributes,
        sal_Int32 nUid )
    {
        if ( xAttributes.is() )
        {
            OUString aValue( xAttributes->getValueByUidName( nUid, rAttrName ) );
            if ( !aValue.isEmpty() )
            {
                if ( aValue == "true" )
                {
                    *pRet = sal_True;
                    return true;
                }
                else if ( aValue == "false" )
                {
                    *pRet = sal_False;
                    return true;
                }
                else
                {
                    throw xml::sax::SAXException(rAttrName + ": no boolean value (true|false)!", Reference< XInterface >(), Any() );
                }
            }
        }
        return false;
    }

    // -----------------------------------------------------------------------------
    // XElement
    // -----------------------------------------------------------------------------

    Reference< xml::input::XElement > BasicElementBase::getParent()
        throw (RuntimeException)
    {
        return static_cast< xml::input::XElement* >( m_pParent );
    }

    // -----------------------------------------------------------------------------

    OUString BasicElementBase::getLocalName()
        throw (RuntimeException)
    {
        return m_aLocalName;
    }

    // -----------------------------------------------------------------------------

    sal_Int32 BasicElementBase::getUid()
        throw (RuntimeException)
    {
        sal_Int32 nId = -1;
        if ( m_pImport )
            nId = m_pImport->XMLNS_UID;
        return nId;
    }

    // -----------------------------------------------------------------------------

    Reference< xml::input::XAttributes > BasicElementBase::getAttributes()
        throw (RuntimeException)
    {
        return m_xAttributes;
    }

    // -----------------------------------------------------------------------------

    Reference< xml::input::XElement > BasicElementBase::startChildElement(
        sal_Int32 /*nUid*/, const OUString& /*rLocalName*/,
        const Reference< xml::input::XAttributes >& /*xAttributes*/ )
        throw (xml::sax::SAXException, RuntimeException)
    {
        throw xml::sax::SAXException("unexpected element!", Reference< XInterface >(), Any() );
    }

    // -----------------------------------------------------------------------------

void BasicElementBase::characters( const OUString& /*rChars*/ )
        throw (xml::sax::SAXException, RuntimeException)
    {
        // not used, all characters ignored
    }

    // -----------------------------------------------------------------------------

void BasicElementBase::ignorableWhitespace( const OUString& /*rWhitespaces*/ )
        throw (xml::sax::SAXException, RuntimeException)
    {
    }

    // -----------------------------------------------------------------------------

void BasicElementBase::processingInstruction( const OUString& /*rTarget*/, const OUString& /*rData*/ )
        throw (xml::sax::SAXException, RuntimeException)
    {
    }

    // -----------------------------------------------------------------------------

    void BasicElementBase::endElement()
        throw (xml::sax::SAXException, RuntimeException)
    {
    }


    // =============================================================================
    // BasicLibrariesElement
    // =============================================================================

    BasicLibrariesElement::BasicLibrariesElement( const OUString& rLocalName,
            const Reference< xml::input::XAttributes >& xAttributes,
            BasicElementBase* pParent, BasicImport* pImport,
            const Reference< script::XLibraryContainer2 >& rxLibContainer )
        :BasicElementBase( rLocalName, xAttributes, pParent, pImport )
        ,m_xLibContainer( rxLibContainer )
    {
    }

    // -----------------------------------------------------------------------------
    // XElement
    // -----------------------------------------------------------------------------

    Reference< xml::input::XElement > BasicLibrariesElement::startChildElement(
            sal_Int32 nUid, const OUString& rLocalName,
            const Reference< xml::input::XAttributes >& xAttributes )
        throw (xml::sax::SAXException, RuntimeException)
    {
        Reference< xml::input::XElement > xElement;

        if ( nUid != m_pImport->XMLNS_UID )
        {
            throw xml::sax::SAXException( "illegal namespace!", Reference< XInterface >(), Any() );
        }
        else if ( rLocalName == "library-linked" )
        {
            if ( xAttributes.is() )
            {
                OUString aName = xAttributes->getValueByUidName( m_pImport->XMLNS_UID, "name" );

                OUString aStorageURL = xAttributes->getValueByUidName(m_pImport->XMLNS_XLINK_UID, "href" );

                sal_Bool bReadOnly = sal_False;
                getBoolAttr( &bReadOnly,"readonly", xAttributes, m_pImport->XMLNS_UID );

                if ( m_xLibContainer.is() )
                {
                    try
                    {
                        Reference< container::XNameAccess > xLib(
                            m_xLibContainer->createLibraryLink( aName, aStorageURL, bReadOnly ) );
                        if ( xLib.is() )
                            xElement.set( new BasicElementBase( rLocalName, xAttributes, this, m_pImport ) );
                    }
                    catch ( const container::ElementExistException& e )
                    {
                        OSL_TRACE( "BasicLibrariesElement::startChildElement: caught ElementExceptionExist reason %s",
                            OUStringToOString( e.Message, RTL_TEXTENCODING_ASCII_US ).pData->buffer );
                    }
                    catch ( const lang::IllegalArgumentException& e )
                    {
                        OSL_TRACE( "BasicLibrariesElement::startChildElement: caught IllegalArgumentException reason %s",
                            OUStringToOString( e.Message, RTL_TEXTENCODING_ASCII_US ).pData->buffer );
                    }
                }
            }
        }
        else if ( rLocalName == "library-embedded" )
        {
            // TODO: create password protected libraries

            if ( xAttributes.is() )
            {
                OUString aName = xAttributes->getValueByUidName( m_pImport->XMLNS_UID, "name" );

                sal_Bool bReadOnly = sal_False;
                getBoolAttr( &bReadOnly, "readonly", xAttributes, m_pImport->XMLNS_UID );

                if ( m_xLibContainer.is() )
                {
                    try
                    {
                        Reference< container::XNameContainer > xLib;
                        if ( m_xLibContainer->hasByName( aName ) )
                        {
                            // Standard library
                            m_xLibContainer->getByName( aName ) >>= xLib;
                        }
                        else
                        {
                            xLib.set( m_xLibContainer->createLibrary( aName ) );
                        }

                        if ( xLib.is() )
                            xElement.set( new BasicEmbeddedLibraryElement( rLocalName, xAttributes, this, m_pImport, m_xLibContainer, aName, bReadOnly ) );
                    }
                    catch ( const lang::IllegalArgumentException& e )
                    {
                        OSL_TRACE( "BasicLibrariesElement::startChildElement: caught IllegalArgumentException reason %s",
                            OUStringToOString( e.Message, RTL_TEXTENCODING_ASCII_US ).pData->buffer );
                    }
                }
            }
        }
        else
        {
            throw xml::sax::SAXException( "expected library-linked or library-embedded element!", Reference< XInterface >(), Any() );
        }

        return xElement;
    }

    // -----------------------------------------------------------------------------

    void BasicLibrariesElement::endElement()
        throw (xml::sax::SAXException, RuntimeException)
    {
    }


    // =============================================================================
    // BasicEmbeddedLibraryElement
    // =============================================================================

    BasicEmbeddedLibraryElement::BasicEmbeddedLibraryElement( const OUString& rLocalName,
            const Reference< xml::input::XAttributes >& xAttributes,
            BasicElementBase* pParent, BasicImport* pImport,
            const Reference< script::XLibraryContainer2 >& rxLibContainer,
            const OUString& rLibName, bool bReadOnly )
        :BasicElementBase( rLocalName, xAttributes, pParent, pImport )
        ,m_xLibContainer( rxLibContainer )
        ,m_aLibName( rLibName )
        ,m_bReadOnly( bReadOnly )
    {
        try
        {
            if ( m_xLibContainer.is() && m_xLibContainer->hasByName( m_aLibName ) )
                m_xLibContainer->getByName( m_aLibName ) >>= m_xLib;
        }
        catch ( const lang::WrappedTargetException& e )
        {
            OSL_TRACE( "BasicEmbeddedLibraryElement CTOR: caught WrappedTargetException reason %s",
                OUStringToOString( e.Message, RTL_TEXTENCODING_ASCII_US ).pData->buffer );
        }
    }

    // -----------------------------------------------------------------------------
    // XElement
    // -----------------------------------------------------------------------------

    Reference< xml::input::XElement > BasicEmbeddedLibraryElement::startChildElement(
            sal_Int32 nUid, const OUString& rLocalName,
            const Reference< xml::input::XAttributes >& xAttributes )
        throw (xml::sax::SAXException, RuntimeException)
    {
        Reference< xml::input::XElement > xElement;

        if ( nUid != m_pImport->XMLNS_UID )
        {
            throw xml::sax::SAXException( "illegal namespace!", Reference< XInterface >(), Any() );
        }
        else if ( rLocalName == "module" )
        {
            if ( xAttributes.is() )
            {
                OUString aName = xAttributes->getValueByUidName(m_pImport->XMLNS_UID, "name" );

                if ( m_xLib.is() && !aName.isEmpty() )
                    xElement.set( new BasicModuleElement( rLocalName, xAttributes, this, m_pImport, m_xLib, aName ) );
            }
        }
        else
        {
            throw xml::sax::SAXException( "expected module element!", Reference< XInterface >(), Any() );
        }

        return xElement;
    }

    // -----------------------------------------------------------------------------

    void BasicEmbeddedLibraryElement::endElement()
        throw (xml::sax::SAXException, RuntimeException)
    {
        if ( m_xLibContainer.is() && m_xLibContainer->hasByName( m_aLibName ) && m_bReadOnly )
            m_xLibContainer->setLibraryReadOnly( m_aLibName, m_bReadOnly );
    }


    // =============================================================================
    // BasicModuleElement
    // =============================================================================

    BasicModuleElement::BasicModuleElement( const OUString& rLocalName,
            const Reference< xml::input::XAttributes >& xAttributes,
            BasicElementBase* pParent, BasicImport* pImport,
            const Reference< container::XNameContainer >& rxLib, const OUString& rName )
        :BasicElementBase( rLocalName, xAttributes, pParent, pImport )
        ,m_xLib( rxLib )
        ,m_aName( rName )
    {
    }

    // -----------------------------------------------------------------------------
    // XElement
    // -----------------------------------------------------------------------------

    Reference< xml::input::XElement > BasicModuleElement::startChildElement(
            sal_Int32 nUid, const OUString& rLocalName,
            const Reference< xml::input::XAttributes >& xAttributes )
        throw (xml::sax::SAXException, RuntimeException)
    {
        // TODO: <byte-code>

        Reference< xml::input::XElement > xElement;

        if ( nUid != m_pImport->XMLNS_UID )
        {
            throw xml::sax::SAXException( "illegal namespace!", Reference< XInterface >(), Any() );
        }
        else if ( rLocalName == "source-code" )
        {
            // TODO: password protected libraries

            if ( xAttributes.is() )
            {
                if ( m_xLib.is() && !m_aName.isEmpty() )
                    xElement.set( new BasicSourceCodeElement( rLocalName, xAttributes, this, m_pImport, m_xLib, m_aName ) );
            }
        }
        else
        {
            throw xml::sax::SAXException( "expected source-code element!", Reference< XInterface >(), Any() );
        }

        return xElement;
    }

    // -----------------------------------------------------------------------------

    void BasicModuleElement::endElement()
        throw (xml::sax::SAXException, RuntimeException)
    {
    }


    // =============================================================================
    // BasicSourceCodeElement
    // =============================================================================

    BasicSourceCodeElement::BasicSourceCodeElement( const OUString& rLocalName,
            const Reference< xml::input::XAttributes >& xAttributes,
            BasicElementBase* pParent, BasicImport* pImport,
            const Reference< container::XNameContainer >& rxLib, const OUString& rName )
        :BasicElementBase( rLocalName, xAttributes, pParent, pImport )
        ,m_xLib( rxLib )
        ,m_aName( rName )
    {
    }

    // -----------------------------------------------------------------------------
    // XElement
    // -----------------------------------------------------------------------------

    void BasicSourceCodeElement::characters( const OUString& rChars )
        throw (xml::sax::SAXException, RuntimeException)
    {
        m_aBuffer.append( rChars );
    }

    // -----------------------------------------------------------------------------

    void BasicSourceCodeElement::endElement()
        throw (xml::sax::SAXException, RuntimeException)
    {
        try
        {
            if ( m_xLib.is() && !m_aName.isEmpty() )
            {
                Any aElement;
                aElement <<= m_aBuffer.makeStringAndClear();
                m_xLib->insertByName( m_aName, aElement );
            }
        }
        catch ( const container::ElementExistException& e )
        {
            OSL_TRACE( "BasicSourceCodeElement::endElement: caught ElementExceptionExist reason %s",
                OUStringToOString( e.Message, RTL_TEXTENCODING_ASCII_US ).pData->buffer );
        }
        catch ( const lang::IllegalArgumentException& e )
        {
            OSL_TRACE( "BasicSourceCodeElement::endElement: caught IllegalArgumentException reason %s",
                OUStringToOString( e.Message, RTL_TEXTENCODING_ASCII_US ).pData->buffer );
        }
        catch ( const lang::WrappedTargetException& e )
        {
            OSL_TRACE( "BasicSourceCodeElement::endElement: caught WrappedTargetException reason %s",
                OUStringToOString( e.Message, RTL_TEXTENCODING_ASCII_US ).pData->buffer );
        }
    }


    // =============================================================================
    // BasicImport
    // =============================================================================

    BasicImport::BasicImport( const Reference< frame::XModel >& rxModel, sal_Bool bOasis )
        :m_xModel( rxModel )
        ,m_bOasis( bOasis )
    {
    }

    // -----------------------------------------------------------------------------

    BasicImport::~BasicImport()
    {
    }

    // -----------------------------------------------------------------------------
    // XRoot
    // -----------------------------------------------------------------------------

    void BasicImport::startDocument( const Reference< xml::input::XNamespaceMapping >& xNamespaceMapping )
        throw (xml::sax::SAXException, RuntimeException)
    {
        if ( xNamespaceMapping.is() )
        {
            OUString aURI;
            if ( m_bOasis )
                aURI = XMLNS_OOO_URI;
            else
                aURI = XMLNS_SCRIPT_URI;
            XMLNS_UID = xNamespaceMapping->getUidByUri( aURI );
            XMLNS_XLINK_UID = xNamespaceMapping->getUidByUri( XMLNS_XLINK_URI );
        }
    }

    // -----------------------------------------------------------------------------

    void BasicImport::endDocument()
        throw (xml::sax::SAXException, RuntimeException)
    {
    }

    // -----------------------------------------------------------------------------

void BasicImport::processingInstruction( const OUString& /*rTarget*/, const OUString& /*rData*/ )
        throw (xml::sax::SAXException, RuntimeException)
    {
    }

    // -----------------------------------------------------------------------------

void BasicImport::setDocumentLocator( const Reference< xml::sax::XLocator >& /*xLocator*/ )
        throw (xml::sax::SAXException, RuntimeException)
    {
    }

    // -----------------------------------------------------------------------------

    Reference< xml::input::XElement > BasicImport::startRootElement( sal_Int32 nUid, const OUString& rLocalName,
            Reference< xml::input::XAttributes > const & xAttributes )
        throw (xml::sax::SAXException, RuntimeException)
    {
        Reference< xml::input::XElement > xElement;

        if ( nUid != XMLNS_UID )
        {
            throw xml::sax::SAXException( "illegal namespace!", Reference< XInterface >(), Any() );
        }
        else if ( rLocalName == "libraries" )
        {
            Reference< script::XLibraryContainer2 > xLibContainer;

            // try the XEmbeddedScripts interface
            Reference< document::XEmbeddedScripts > xDocumentScripts( m_xModel, UNO_QUERY );
            if ( xDocumentScripts.is() )
                xLibContainer.set( xDocumentScripts->getBasicLibraries().get() );

            if ( !xLibContainer.is() )
            {
                // try the "BasicLibraries" property (old-style, for compatibility)
                Reference< beans::XPropertySet > xPSet( m_xModel, UNO_QUERY );
                if ( xPSet.is() )
                    xPSet->getPropertyValue("BasicLibraries" ) >>= xLibContainer;
            }

            OSL_ENSURE( xLibContainer.is(), "BasicImport::startRootElement: nowhere to import to!" );

            if ( xLibContainer.is() )
            {
                xElement.set( new BasicLibrariesElement( rLocalName, xAttributes, 0, this, xLibContainer ) );
            }
        }
        else
        {
            throw xml::sax::SAXException("illegal root element (expected libraries) given: " + rLocalName, Reference< XInterface >(), Any() );
        }

        return xElement;
    }


    // =============================================================================
    // component operations
    // =============================================================================

    OUString getImplementationName_XMLBasicImporter()
    {
        static OUString* pImplName = 0;
        if ( !pImplName )
        {
            ::osl::MutexGuard aGuard( ::osl::Mutex::getGlobalMutex() );
            if ( !pImplName )
            {
                static OUString aImplName( "com.sun.star.comp.xmlscript.XMLBasicImporter" );
                pImplName = &aImplName;
            }
        }
        return *pImplName;
    }

    // -----------------------------------------------------------------------------

    Sequence< OUString > getSupportedServiceNames_XMLBasicImporter()
    {
        static Sequence< OUString >* pNames = 0;
        if ( !pNames )
        {
            ::osl::MutexGuard aGuard( ::osl::Mutex::getGlobalMutex() );
            if ( !pNames )
            {
                static Sequence< OUString > aNames(1);
                aNames.getArray()[0] = "com.sun.star.document.XMLBasicImporter";
                pNames = &aNames;
            }
        }
        return *pNames;
    }

    // -----------------------------------------------------------------------------

    OUString getImplementationName_XMLOasisBasicImporter()
    {
        static OUString* pImplName = 0;
        if ( !pImplName )
        {
            ::osl::MutexGuard aGuard( ::osl::Mutex::getGlobalMutex() );
            if ( !pImplName )
            {
                static OUString aImplName( "com.sun.star.comp.xmlscript.XMLOasisBasicImporter" );
                pImplName = &aImplName;
            }
        }
        return *pImplName;
    }

    // -----------------------------------------------------------------------------

    Sequence< OUString > getSupportedServiceNames_XMLOasisBasicImporter()
    {
        static Sequence< OUString >* pNames = 0;
        if ( !pNames )
        {
            ::osl::MutexGuard aGuard( ::osl::Mutex::getGlobalMutex() );
            if ( !pNames )
            {
                static Sequence< OUString > aNames(1);
                aNames.getArray()[0] = "com.sun.star.document.XMLOasisBasicImporter";
                pNames = &aNames;
            }
        }
        return *pNames;
    }


    // =============================================================================
    // XMLBasicImporterBase
    // =============================================================================

    XMLBasicImporterBase::XMLBasicImporterBase( const Reference< XComponentContext >& rxContext, sal_Bool bOasis )
        :m_xContext( rxContext )
        ,m_bOasis( bOasis )
    {
    }

    // -----------------------------------------------------------------------------

    XMLBasicImporterBase::~XMLBasicImporterBase()
    {
    }

    // -----------------------------------------------------------------------------
    // XServiceInfo
    // -----------------------------------------------------------------------------

    sal_Bool XMLBasicImporterBase::supportsService( const OUString& rServiceName ) throw (RuntimeException)
    {
        Sequence< OUString > aNames( getSupportedServiceNames() );
        const OUString* pNames = aNames.getConstArray();
        const OUString* pEnd = pNames + aNames.getLength();
        for ( ; pNames != pEnd && !pNames->equals( rServiceName ); ++pNames )
            ;

        return pNames != pEnd;
    }

    // -----------------------------------------------------------------------------
    // XImporter
    // -----------------------------------------------------------------------------

    void XMLBasicImporterBase::setTargetDocument( const Reference< XComponent >& rxDoc )
        throw (IllegalArgumentException, RuntimeException)
    {
        ::osl::MutexGuard aGuard( m_aMutex );

        m_xModel.set( rxDoc, UNO_QUERY );

        if ( !m_xModel.is() )
        {
            throw IllegalArgumentException( "XMLBasicExporter::setTargetDocument: no document model!", Reference< XInterface >(), 1 );
        }

        if ( m_xContext.is() )
        {
            Reference< XMultiComponentFactory > xSMgr( m_xContext->getServiceManager() );
            if ( xSMgr.is() )
            {
                Reference < xml::input::XRoot > xRoot( new BasicImport( m_xModel, m_bOasis ) );
                Sequence < Any > aArgs( 1 );
                aArgs[0] <<= xRoot;
                m_xHandler.set( xSMgr->createInstanceWithArgumentsAndContext("com.sun.star.xml.input.SaxDocumentHandler", aArgs, m_xContext ), UNO_QUERY );
            }
        }
    }

    // -----------------------------------------------------------------------------
    // XDocumentHandler
    // -----------------------------------------------------------------------------

    void XMLBasicImporterBase::startDocument()
        throw (xml::sax::SAXException, RuntimeException)
    {
        ::osl::MutexGuard aGuard( m_aMutex );

        if ( m_xHandler.is() )
            m_xHandler->startDocument();
    }

    // -----------------------------------------------------------------------------

    void XMLBasicImporterBase::endDocument()
        throw (xml::sax::SAXException, RuntimeException)
    {
        ::osl::MutexGuard aGuard( m_aMutex );

        if ( m_xHandler.is() )
            m_xHandler->endDocument();
    }

    // -----------------------------------------------------------------------------

    void XMLBasicImporterBase::startElement( const OUString& aName,
            const Reference< xml::sax::XAttributeList >& xAttribs )
        throw (xml::sax::SAXException, RuntimeException)
    {
        ::osl::MutexGuard aGuard( m_aMutex );

        if ( m_xHandler.is() )
            m_xHandler->startElement( aName, xAttribs );
    }

    // -----------------------------------------------------------------------------

    void XMLBasicImporterBase::endElement( const OUString& aName )
        throw (xml::sax::SAXException, RuntimeException)
    {
        ::osl::MutexGuard aGuard( m_aMutex );

        if ( m_xHandler.is() )
            m_xHandler->endElement( aName );
    }

    // -----------------------------------------------------------------------------

    void XMLBasicImporterBase::characters( const OUString& aChars )
        throw (xml::sax::SAXException, RuntimeException)
    {
        ::osl::MutexGuard aGuard( m_aMutex );

        if ( m_xHandler.is() )
            m_xHandler->characters( aChars );
    }

    // -----------------------------------------------------------------------------

    void XMLBasicImporterBase::ignorableWhitespace( const OUString& aWhitespaces )
        throw (xml::sax::SAXException, RuntimeException)
    {
        ::osl::MutexGuard aGuard( m_aMutex );

        if ( m_xHandler.is() )
            m_xHandler->ignorableWhitespace( aWhitespaces );
    }

    // -----------------------------------------------------------------------------

    void XMLBasicImporterBase::processingInstruction( const OUString& aTarget,
            const OUString& aData )
        throw (xml::sax::SAXException, RuntimeException)
    {
        ::osl::MutexGuard aGuard( m_aMutex );

        if ( m_xHandler.is() )
            m_xHandler->processingInstruction( aTarget, aData );
    }

    // -----------------------------------------------------------------------------

    void XMLBasicImporterBase::setDocumentLocator( const Reference< xml::sax::XLocator >& xLocator )
        throw (xml::sax::SAXException, RuntimeException)
    {
        ::osl::MutexGuard aGuard( m_aMutex );

        if ( m_xHandler.is() )
            m_xHandler->setDocumentLocator( xLocator );
    }


    // =============================================================================
    // XMLBasicImporter
    // =============================================================================

    XMLBasicImporter::XMLBasicImporter( const Reference< XComponentContext >& rxContext )
        :XMLBasicImporterBase( rxContext, sal_False )
    {
    }

    // -----------------------------------------------------------------------------

    XMLBasicImporter::~XMLBasicImporter()
    {
    }

    // -----------------------------------------------------------------------------
    // XServiceInfo
    // -----------------------------------------------------------------------------

    OUString XMLBasicImporter::getImplementationName(  ) throw (RuntimeException)
    {
        return getImplementationName_XMLBasicImporter();
    }

    // -----------------------------------------------------------------------------

    Sequence< OUString > XMLBasicImporter::getSupportedServiceNames(  ) throw (RuntimeException)
    {
        return getSupportedServiceNames_XMLBasicImporter();
    }


    // =============================================================================
    // XMLOasisBasicImporter
    // =============================================================================

    XMLOasisBasicImporter::XMLOasisBasicImporter( const Reference< XComponentContext >& rxContext )
        :XMLBasicImporterBase( rxContext, sal_True )
    {
    }

    // -----------------------------------------------------------------------------

    XMLOasisBasicImporter::~XMLOasisBasicImporter()
    {
    }

    // -----------------------------------------------------------------------------
    // XServiceInfo
    // -----------------------------------------------------------------------------

    OUString XMLOasisBasicImporter::getImplementationName(  ) throw (RuntimeException)
    {
        return getImplementationName_XMLOasisBasicImporter();
    }

    // -----------------------------------------------------------------------------

    Sequence< OUString > XMLOasisBasicImporter::getSupportedServiceNames(  ) throw (RuntimeException)
    {
        return getSupportedServiceNames_XMLOasisBasicImporter();
    }


    // =============================================================================
    // component operations
    // =============================================================================

    Reference< XInterface > SAL_CALL create_XMLBasicImporter(
        Reference< XComponentContext > const & xContext )
        SAL_THROW(())
    {
        return static_cast< lang::XTypeProvider * >( new XMLBasicImporter( xContext ) );
    }

    // -----------------------------------------------------------------------------

    Reference< XInterface > SAL_CALL create_XMLOasisBasicImporter(
        Reference< XComponentContext > const & xContext )
        SAL_THROW(())
    {
        return static_cast< lang::XTypeProvider * >( new XMLOasisBasicImporter( xContext ) );
    }

    // -----------------------------------------------------------------------------

//.........................................................................
}   // namespace xmlscript
//.........................................................................

/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
